<?php
/**
 * Team Corrections Manager
 * Manages manual corrections for team names and logos
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class ApiFootballTeamCorrections {
    
    private $table_name;
    
    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'api_football_team_corrections';
    }
    
    /**
     * Get team correction by team ID and optionally league ID
     */
    public function getTeamCorrection($team_id, $league_id = null) {
        global $wpdb;
        
        $query = "SELECT * FROM {$this->table_name} WHERE team_id = %d";
        $params = array($team_id);
        
        if ($league_id !== null) {
            $query .= " AND (league_id = %d OR league_id IS NULL)";
            $params[] = $league_id;
            $query .= " ORDER BY league_id DESC"; // Prefer league-specific corrections
        }
        
        $query .= " LIMIT 1";
        
        return $wpdb->get_row($wpdb->prepare($query, $params), ARRAY_A);
    }
    
    /**
     * Get all team corrections
     */
    public function getAllCorrections($league_id = null) {
        global $wpdb;
        
        $query = "SELECT * FROM {$this->table_name}";
        $params = array();
        
        if ($league_id !== null) {
            $query .= " WHERE league_id = %d OR league_id IS NULL";
            $params[] = $league_id;
        }
        
        $query .= " ORDER BY corrected_name ASC, original_name ASC";
        
        if (!empty($params)) {
            $query = $wpdb->prepare($query, $params);
        }
        
        return $wpdb->get_results($query, ARRAY_A);
    }
    
    /**
     * Save team correction
     */
    public function saveCorrection($data) {
        global $wpdb;
        
        $defaults = array(
            'team_id' => 0,
            'original_name' => '',
            'corrected_name' => null,
            'custom_logo_url' => null,
            'league_id' => null
        );
        
        $data = wp_parse_args($data, $defaults);
        
        // Check if correction already exists
        $existing = $this->getTeamCorrection($data['team_id'], $data['league_id']);
        
        if ($existing) {
            // Update existing
            return $wpdb->update(
                $this->table_name,
                array(
                    'original_name' => sanitize_text_field($data['original_name']),
                    'corrected_name' => !empty($data['corrected_name']) ? sanitize_text_field($data['corrected_name']) : null,
                    'custom_logo_url' => !empty($data['custom_logo_url']) ? esc_url_raw($data['custom_logo_url']) : null,
                    'league_id' => !empty($data['league_id']) ? intval($data['league_id']) : null
                ),
                array('id' => $existing['id']),
                array('%s', '%s', '%s', '%d'),
                array('%d')
            );
        } else {
            // Insert new
            return $wpdb->insert(
                $this->table_name,
                array(
                    'team_id' => intval($data['team_id']),
                    'original_name' => sanitize_text_field($data['original_name']),
                    'corrected_name' => !empty($data['corrected_name']) ? sanitize_text_field($data['corrected_name']) : null,
                    'custom_logo_url' => !empty($data['custom_logo_url']) ? esc_url_raw($data['custom_logo_url']) : null,
                    'league_id' => !empty($data['league_id']) ? intval($data['league_id']) : null
                ),
                array('%d', '%s', '%s', '%s', '%d')
            );
        }
    }
    
    /**
     * Delete team correction
     */
    public function deleteCorrection($id) {
        global $wpdb;
        
        return $wpdb->delete(
            $this->table_name,
            array('id' => intval($id)),
            array('%d')
        );
    }
    
    /**
     * Apply corrections to team data
     */
    public function applyCorrections($team_data, $league_id = null) {
        if (!is_array($team_data) || !isset($team_data['id'])) {
            return $team_data;
        }
        
        $correction = $this->getTeamCorrection($team_data['id'], $league_id);
        
        if ($correction) {
            // Apply name correction
            if (!empty($correction['corrected_name'])) {
                $team_data['name'] = $correction['corrected_name'];
            }
            
            // Apply logo correction
            if (!empty($correction['custom_logo_url'])) {
                $team_data['logo'] = $correction['custom_logo_url'];
            }
        }
        
        return $team_data;
    }
    
    /**
     * Apply corrections to multiple teams
     */
    public function applyCorrectionsToTeams($teams, $league_id = null) {
        if (!is_array($teams)) {
            return $teams;
        }
        
        foreach ($teams as &$team) {
            $team = $this->applyCorrections($team, $league_id);
        }
        
        return $teams;
    }
    
    /**
     * Import corrections from JSON
     */
    public function importCorrections($json_data) {
        $corrections = json_decode($json_data, true);
        
        if (!is_array($corrections)) {
            return array('success' => false, 'message' => __('Invalid JSON data', 'api-football'));
        }
        
        $imported = 0;
        $failed = 0;
        
        foreach ($corrections as $correction) {
            if ($this->saveCorrection($correction)) {
                $imported++;
            } else {
                $failed++;
            }
        }
        
        return array(
            'success' => true,
            'message' => sprintf(
                __('Import complete. %d corrections imported, %d failed.', 'api-football'),
                $imported,
                $failed
            )
        );
    }
    
    /**
     * Export corrections to JSON
     */
    public function exportCorrections() {
        $corrections = $this->getAllCorrections();
        
        // Remove unnecessary fields
        foreach ($corrections as &$correction) {
            unset($correction['id']);
            unset($correction['created_at']);
            unset($correction['updated_at']);
        }
        
        return json_encode($corrections, JSON_PRETTY_PRINT);
    }
    
    /**
     * Get teams from API data that might need corrections
     */
    public function getTeamsFromApiData($api_data) {
        $teams = array();
        
        // Extract teams from standings data
        if (isset($api_data['response'][0]['league']['standings'][0])) {
            foreach ($api_data['response'][0]['league']['standings'][0] as $standing) {
                if (isset($standing['team'])) {
                    $teams[$standing['team']['id']] = array(
                        'id' => $standing['team']['id'],
                        'name' => $standing['team']['name'],
                        'logo' => $standing['team']['logo']
                    );
                }
            }
        }
        
        // Extract teams from fixtures data
        if (isset($api_data['response']) && is_array($api_data['response'])) {
            foreach ($api_data['response'] as $fixture) {
                if (isset($fixture['teams']['home'])) {
                    $teams[$fixture['teams']['home']['id']] = array(
                        'id' => $fixture['teams']['home']['id'],
                        'name' => $fixture['teams']['home']['name'],
                        'logo' => $fixture['teams']['home']['logo']
                    );
                }
                if (isset($fixture['teams']['away'])) {
                    $teams[$fixture['teams']['away']['id']] = array(
                        'id' => $fixture['teams']['away']['id'],
                        'name' => $fixture['teams']['away']['name'],
                        'logo' => $fixture['teams']['away']['logo']
                    );
                }
            }
        }
        
        return array_values($teams);
    }
}