<?php
/**
 * API Football Shortcodes
 * Handles all shortcode functionality
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class ApiFootballShortcodes {
    
    private $api_client;
    
    public function __construct() {
        $this->api_client = new ApiFootballClient();
        $this->registerShortcodes();
    }
    
    /**
     * Register all shortcodes
     */
    private function registerShortcodes() {
        add_shortcode('api_football_standings', array($this, 'standingsShortcode'));
        add_shortcode('api_football_standings_full', array($this, 'standingsFullShortcode'));
        add_shortcode('api_football_fixtures', array($this, 'fixturesShortcode'));
        add_shortcode('api_football_results', array($this, 'resultsShortcode'));
        add_shortcode('api_football_schedule', array($this, 'scheduleShortcode'));
        add_shortcode('api_football_results_full', array($this, 'resultsFullShortcode'));
    }
    
    /**
     * Standings shortcode (compact version)
     */
    public function standingsShortcode($atts) {
        $atts = shortcode_atts(array(
            'league' => '',
            'season' => '',
            'limit' => '10',
            'class' => 'api-football-standings'
        ), $atts, 'api_football_standings');
        
        $data = $this->api_client->getStandings($atts['league'], $atts['season']);
        
        if (empty($data['response'])) {
            return '<div class="api-football-error">' . __('No standings data available.', 'api-football') . '</div>';
        }
        
        $standings = $data['response'][0]['league']['standings'][0];
        $limit = intval($atts['limit']);
        
        if ($limit > 0) {
            $standings = array_slice($standings, 0, $limit);
        }
        
        return $this->renderStandingsTable($standings, $atts['class'], 'compact');
    }
    
    /**
     * Full standings shortcode
     */
    public function standingsFullShortcode($atts) {
        $atts = shortcode_atts(array(
            'league' => '',
            'season' => '',
            'class' => 'api-football-standings-full'
        ), $atts, 'api_football_standings_full');
        
        $data = $this->api_client->getStandings($atts['league'], $atts['season']);
        
        if (empty($data['response'])) {
            return '<div class="api-football-error">' . __('No standings data available.', 'api-football') . '</div>';
        }
        
        $standings = $data['response'][0]['league']['standings'][0];
        
        return $this->renderStandingsTable($standings, $atts['class'], 'full');
    }
    
    /**
     * Fixtures shortcode (upcoming matches)
     */
    public function fixturesShortcode($atts) {
        $atts = shortcode_atts(array(
            'league' => '',
            'season' => '',
            'team' => '',
            'next' => '5',
            'class' => 'api-football-fixtures'
        ), $atts, 'api_football_fixtures');
        
        $data = $this->api_client->getFixtures(
            $atts['league'], 
            $atts['season'], 
            $atts['team'], 
            $atts['next']
        );
        
        if (empty($data['response'])) {
            return '<div class="api-football-error">' . __('No upcoming fixtures available.', 'api-football') . '</div>';
        }
        
        return $this->renderFixturesList($data['response'], $atts['class'], 'upcoming');
    }
    
    /**
     * Results shortcode (past matches)
     */
    public function resultsShortcode($atts) {
        $atts = shortcode_atts(array(
            'league' => '',
            'season' => '',
            'team' => '',
            'last' => '5',
            'class' => 'api-football-results'
        ), $atts, 'api_football_results');
        
        $data = $this->api_client->getFixtures(
            $atts['league'], 
            $atts['season'], 
            $atts['team'], 
            null, 
            $atts['last']
        );
        
        if (empty($data['response'])) {
            return '<div class="api-football-error">' . __('No recent results available.', 'api-football') . '</div>';
        }
        
        return $this->renderFixturesList($data['response'], $atts['class'], 'results');
    }
    
    /**
     * Schedule shortcode (all upcoming matches)
     */
    public function scheduleShortcode($atts) {
        $atts = shortcode_atts(array(
            'league' => '',
            'season' => '',
            'from' => '',
            'to' => '',
            'class' => 'api-football-schedule'
        ), $atts, 'api_football_schedule');
        
        // If no date range specified, get upcoming matches for rest of season
        $from = $atts['from'] ?: date('Y-m-d');
        $to = $atts['to'] ?: date('Y-12-31');
        
        $data = $this->api_client->getFixtures(
            $atts['league'], 
            $atts['season'], 
            null, 
            null, 
            null, 
            $from, 
            $to
        );
        
        if (empty($data['response'])) {
            return '<div class="api-football-error">' . __('No scheduled matches available.', 'api-football') . '</div>';
        }
        
        return $this->renderScheduleTable($data['response'], $atts['class']);
    }
    
    /**
     * Full results shortcode (all past matches)
     */
    public function resultsFullShortcode($atts) {
        $atts = shortcode_atts(array(
            'league' => '',
            'season' => '',
            'from' => '',
            'to' => '',
            'class' => 'api-football-results-full'
        ), $atts, 'api_football_results_full');
        
        // If no date range specified, get matches from season start to now
        $from = $atts['from'] ?: ($atts['season'] ?: date('Y')) . '-01-01';
        $to = $atts['to'] ?: date('Y-m-d');
        
        $data = $this->api_client->getFixtures(
            $atts['league'], 
            $atts['season'], 
            null, 
            null, 
            null, 
            $from, 
            $to
        );
        
        if (empty($data['response'])) {
            return '<div class="api-football-error">' . __('No match results available.', 'api-football') . '</div>';
        }
        
        // Filter only finished matches
        $finished_matches = array_filter($data['response'], function($match) {
            return $match['fixture']['status']['short'] === 'FT';
        });
        
        return $this->renderResultsTable($finished_matches, $atts['class']);
    }
    
    /**
     * Render standings table
     */
    private function renderStandingsTable($standings, $class, $mode = 'compact') {
        ob_start();
        ?>
        <div class="<?php echo esc_attr($class); ?> api-football-widget">
            <table class="api-football-table standings-table">
                <thead>
                    <tr>
                        <th class="rank"><?php _e('#', 'api-football'); ?></th>
                        <th class="team"><?php _e('Team', 'api-football'); ?></th>
                        <th class="played"><?php _e('P', 'api-football'); ?></th>
                        <th class="wins"><?php _e('W', 'api-football'); ?></th>
                        <th class="draws"><?php _e('D', 'api-football'); ?></th>
                        <th class="losses"><?php _e('L', 'api-football'); ?></th>
                        <th class="goal-diff"><?php _e('GD', 'api-football'); ?></th>
                        <th class="points"><?php _e('Pts', 'api-football'); ?></th>
                        <?php if ($mode === 'full'): ?>
                        <th class="form"><?php _e('Form', 'api-football'); ?></th>
                        <?php endif; ?>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($standings as $team): 
                        $row_class = $this->getRowClass($team['rank']);
                        $logo_url = $this->api_client->cacheTeamLogo($team['team']['logo'], $team['team']['id']);
                    ?>
                    <tr class="<?php echo esc_attr($row_class); ?>">
                        <td class="rank" data-label="<?php _e('#', 'api-football'); ?>"><?php echo esc_html($team['rank']); ?></td>
                        <td class="team" data-label="<?php _e('Team', 'api-football'); ?>">
							<div class="team-name">
                            	<img src="<?php echo esc_url($logo_url); ?>" alt="<?php echo esc_attr($team['team']['name']); ?>" class="team-logo">
                            	<span class="team-name"><?php echo esc_html($team['team']['name']); ?></span>
							</div>
                        </td>
                        <td class="played" data-label="<?php _e('P', 'api-football'); ?>"><?php echo esc_html($team['all']['played']); ?></td>
                        <td class="wins" data-label="<?php _e('W', 'api-football'); ?>"><?php echo esc_html($team['all']['win']); ?></td>
                        <td class="draws" data-label="<?php _e('D', 'api-football'); ?>"><?php echo esc_html($team['all']['draw']); ?></td>
                        <td class="losses" data-label="<?php _e('L', 'api-football'); ?>"><?php echo esc_html($team['all']['lose']); ?></td>
                        <td class="goal-diff" data-label="<?php _e('GD', 'api-football'); ?>"><?php echo esc_html($team['goalsDiff']); ?></td>
                        <td class="points" data-label="<?php _e('Pts', 'api-football'); ?>"><?php echo esc_html($team['points']); ?></td>
                        <?php if ($mode === 'full'): ?>
                        <td class="form" data-label="<?php _e('Form', 'api-football'); ?>">
                            <span class="form-string"><?php echo esc_html($team['form']); ?></span>
                        </td>
                        <?php endif; ?>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Render fixtures list
     */
    private function renderFixturesList($fixtures, $class, $type = 'upcoming') {
        ob_start();
        ?>
        <div class="<?php echo esc_attr($class); ?> api-football-widget">
            <div class="fixtures-list">
                <?php foreach ($fixtures as $fixture): 
                    $home_team = $fixture['teams']['home'];
                    $away_team = $fixture['teams']['away'];
                    $home_logo = $this->api_client->cacheTeamLogo($home_team['logo'], $home_team['id']);
                    $away_logo = $this->api_client->cacheTeamLogo($away_team['logo'], $away_team['id']);
                    $date = $this->api_client->formatDate($fixture['fixture']['date']);
                    $time = $this->api_client->formatTime($fixture['fixture']['date']);
                ?>
                <div class="fixture-item">
                    <div class="fixture-date">
                        <span class="date"><?php echo esc_html($date); ?></span>
                        <span class="time"><?php echo esc_html($time); ?></span>
                    </div>
                    <div class="fixture-teams">
                        <div class="home-team">
                            <img src="<?php echo esc_url($home_logo); ?>" alt="<?php echo esc_attr($home_team['name']); ?>" class="team-logo">
                            <span class="team-name"><?php echo esc_html($home_team['name']); ?></span>
                        </div>
                        <div class="match-info">
                            <?php if ($type === 'results' && isset($fixture['goals'])): ?>
                                <span class="score"><?php echo esc_html($fixture['goals']['home'] . ' - ' . $fixture['goals']['away']); ?></span>
                            <?php else: ?>
                                <span class="vs"><?php _e('vs', 'api-football'); ?></span>
                            <?php endif; ?>
                        </div>
                        <div class="away-team">
                            <img src="<?php echo esc_url($away_logo); ?>" alt="<?php echo esc_attr($away_team['name']); ?>" class="team-logo">
                            <span class="team-name"><?php echo esc_html($away_team['name']); ?></span>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Render schedule table
     */
    private function renderScheduleTable($fixtures, $class) {
        ob_start();
        ?>
        <div class="<?php echo esc_attr($class); ?> api-football-widget">
            <table class="api-football-table schedule-table">
                <thead>
                    <tr>
                        <th class="date"><?php _e('Date', 'api-football'); ?></th>
                        <th class="time"><?php _e('Time', 'api-football'); ?></th>
                        <th class="home-team"><?php _e('Home', 'api-football'); ?></th>
                        <th class="away-team"><?php _e('Away', 'api-football'); ?></th>
                        <th class="venue"><?php _e('Venue', 'api-football'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($fixtures as $fixture): 
                        $home_team = $fixture['teams']['home'];
                        $away_team = $fixture['teams']['away'];
                        $home_logo = $this->api_client->cacheTeamLogo($home_team['logo'], $home_team['id']);
                        $away_logo = $this->api_client->cacheTeamLogo($away_team['logo'], $away_team['id']);
                        $date = $this->api_client->formatDate($fixture['fixture']['date']);
                        $time = $this->api_client->formatTime($fixture['fixture']['date']);
                    ?>
                    <tr>
                        <td class="date" data-label="<?php _e('Date', 'api-football'); ?>"><?php echo esc_html($date); ?></td>
                        <td class="time" data-label="<?php _e('Time', 'api-football'); ?>"><?php echo esc_html($time); ?></td>
                        <td class="home-team" data-label="<?php _e('Home', 'api-football'); ?>">
							<div class="team-name">
                            	<img src="<?php echo esc_url($home_logo); ?>" alt="<?php echo esc_attr($home_team['name']); ?>" class="team-logo">
                            	<span class="team-name"><?php echo esc_html($home_team['name']); ?></span>
							</div>
                        </td>
                        <td class="away-team" data-label="<?php _e('Away', 'api-football'); ?>">
							<div class="team-name">
                            	<img src="<?php echo esc_url($away_logo); ?>" alt="<?php echo esc_attr($away_team['name']); ?>" class="team-logo">
                            	<span class="team-name"><?php echo esc_html($away_team['name']); ?></span>
							</div>
                        </td>
                        <td class="venue" data-label="<?php _e('Venue', 'api-football'); ?>"><?php echo esc_html($fixture['fixture']['venue']['name'] ?? ''); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Render results table
     */
    private function renderResultsTable($results, $class) {
        ob_start();
        ?>
        <div class="<?php echo esc_attr($class); ?> api-football-widget">
            <table class="api-football-table results-table">
                <thead>
                    <tr>
                        <th class="date"><?php _e('Date', 'api-football'); ?></th>
                        <th class="home-team"><?php _e('Home', 'api-football'); ?></th>
                        <th class="score"><?php _e('Score', 'api-football'); ?></th>
                        <th class="away-team"><?php _e('Away', 'api-football'); ?></th>
                        <th class="venue"><?php _e('Venue', 'api-football'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($results as $result): 
                        $home_team = $result['teams']['home'];
                        $away_team = $result['teams']['away'];
                        $home_logo = $this->api_client->cacheTeamLogo($home_team['logo'], $home_team['id']);
                        $away_logo = $this->api_client->cacheTeamLogo($away_team['logo'], $away_team['id']);
                        $date = $this->api_client->formatDate($result['fixture']['date']);
                        $score = $result['goals']['home'] . ' - ' . $result['goals']['away'];
                    ?>
                    <tr>
                        <td class="date" data-label="<?php _e('Date', 'api-football'); ?>"><?php echo esc_html($date); ?></td>
                        <td class="home-team" data-label="<?php _e('Home', 'api-football'); ?>">
							<div class="team">
                            	<img src="<?php echo esc_url($home_logo); ?>" alt="<?php echo esc_attr($home_team['name']); ?>" class="team-logo">
                            	<span class="team-name"><?php echo esc_html($home_team['name']); ?></span>
							</div>
                        </td>
                        <td class="score" data-label="<?php _e('Score', 'api-football'); ?>"><?php echo esc_html($score); ?></td>
                        <td class="away-team" data-label="<?php _e('Away', 'api-football'); ?>">
							<div class="team">
								<img src="<?php echo esc_url($away_logo); ?>" alt="<?php echo esc_attr($away_team['name']); ?>" class="team-logo">
								<span class="team-name"><?php echo esc_html($away_team['name']); ?></span>
							</div>
                        </td>
                        <td class="venue" data-label="<?php _e('Venue', 'api-football'); ?>"><?php echo esc_html($result['fixture']['venue']['name'] ?? ''); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Get row CSS class based on team position
     */
    private function getRowClass($rank) {
        if ($rank <= 3) {
            return 'position-top';
        } elseif ($rank <= 6) {
            return 'position-europe';
        } elseif ($rank >= 18) {
            return 'position-relegation';
        }
        return 'position-mid';
    }
} 