<?php
/**
 * API Football Client
 * Handles all API communication with API-Football.com
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class ApiFootballClient {
    
    private $api_key;
    private $base_url = 'https://v3.football.api-sports.io';
    private $options;
    private $team_corrections;
    
    public function __construct() {
        $this->options = get_option('api_football_options', array());
        $this->api_key = !empty($this->options['api_key']) ? $this->options['api_key'] : '21520870122bb16d77ddcffa5d31e225';
        $this->team_corrections = new ApiFootballTeamCorrections();
    }
    
    /**
     * Get league standings
     */
    public function getStandings($league_id = null, $season = null) {
        $league_id = $league_id ?: $this->getOption('league_id', '286');
        $season = $season ?: $this->getOption('season', '2025');
        
        $cache_key = 'api_football_standings_' . $league_id . '_' . $season;
        $cached_data = get_transient($cache_key);
        
        if ($cached_data !== false) {
            return $cached_data;
        }
        
        $endpoint = '/standings';
        $params = array(
            'league' => $league_id,
            'season' => $season
            // Note: No timezone needed for standings as you mentioned
        );
        
        $response = $this->makeRequest($endpoint, $params);
        
        if ($response && !empty($response['response'])) {
            // Apply team corrections
            $response = $this->applyTeamCorrectionsToResponse($response, $league_id);
            
            $cache_duration = $this->getOption('cache_duration', 4 * HOUR_IN_SECONDS);
            set_transient($cache_key, $response, $cache_duration);
            return $response;
        }
        
        // Return cached data if available, even if expired
        $expired_cache = get_option('api_football_backup_' . $cache_key);
        return $expired_cache ?: array('response' => array());
    }
    
    /**
     * Get fixtures (upcoming or past matches)
     */
    public function getFixtures($league_id = null, $season = null, $team_id = null, $next = null, $last = null, $from = null, $to = null) {
        $league_id = $league_id ?: $this->getOption('league_id', '286');
        $season = $season ?: $this->getOption('season', '2025');
        $team_id = $team_id ?: $this->getOption('team_id', '12360');
        $timezone = $this->getOption('timezone', 'Europe/Belgrade');
        
        $cache_key = 'api_football_fixtures_' . md5(serialize(array(
            'league' => $league_id,
            'season' => $season,
            'team' => $team_id,
            'next' => $next,
            'last' => $last,
            'from' => $from,
            'to' => $to,
            'timezone' => $timezone  // Include timezone in cache key
        )));
        
        $cached_data = get_transient($cache_key);
        
        if ($cached_data !== false) {
            return $cached_data;
        }
        
        $endpoint = '/fixtures';
        $params = array(
            'league' => $league_id,
            'season' => $season,
            'timezone' => $timezone  // This is correct - Europe/Belgrade format is valid
        );
        
        if ($team_id) {
            $params['team'] = $team_id;
        }
        
        if ($next) {
            $params['next'] = $next;
        }
        
        if ($last) {
            $params['last'] = $last;
        }
        
        if ($from) {
            $params['from'] = $from;
        }
        
        if ($to) {
            $params['to'] = $to;
        }
        
        $response = $this->makeRequest($endpoint, $params);
        
        if ($response && !empty($response['response'])) {
            // Debug: Log the timezone from response
            if (!empty($response['response'][0])) {
                error_log('API Football Fixture Response Timezone: ' . ($response['response'][0]['fixture']['timezone'] ?? 'not set'));
                error_log('API Football Fixture Response Date: ' . ($response['response'][0]['fixture']['date'] ?? 'not set'));
                error_log('API Football Request Timezone Param: ' . $timezone);
            }
            
            // Apply team corrections
            $response = $this->applyTeamCorrectionsToResponse($response, $league_id);
            
            $cache_duration = $this->getOption('cache_duration', 4 * HOUR_IN_SECONDS);
            set_transient($cache_key, $response, $cache_duration);
            return $response;
        }
        
        // Return cached data if available, even if expired
        $expired_cache = get_option('api_football_backup_' . $cache_key);
        return $expired_cache ?: array('response' => array());
    }
    
    /**
     * Make HTTP request to API
     */
    private function makeRequest($endpoint, $params = array()) {
        $url = $this->base_url . $endpoint;
        
        if (!empty($params)) {
            $url .= '?' . http_build_query($params);
        }
        
        // CORRECTED HEADERS - Use x-apisports-key for direct API-Sports endpoint
        $headers = array(
            'x-apisports-key' => $this->api_key  // Changed from X-RapidAPI-Key
            // Removed X-RapidAPI-Host as it's not needed for direct API-Sports
        );
        
        $args = array(
            'headers' => $headers,
            'timeout' => 30,
            'user-agent' => 'WordPress API Football Plugin/' . API_FOOTBALL_VERSION
        );
        
        // Debug logging
        error_log('API Football Request URL: ' . $url);
        error_log('API Football Headers: ' . json_encode($headers));
        
        $response = wp_remote_get($url, $args);
        
        if (is_wp_error($response)) {
            error_log('API Football Error: ' . $response->get_error_message());
            return false;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        if ($response_code !== 200) {
            error_log('API Football HTTP Error: ' . $response_code . ' - ' . $body);
            return false;
        }
        
        $data = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log('API Football JSON Error: ' . json_last_error_msg());
            return false;
        }
        
        // Store backup copy for fallback
        if (!empty($data['response'])) {
            $cache_key = 'api_football_backup_' . md5($url);
            update_option($cache_key, $data, false);
        }
        
        return $data;
    }
    
    /**
     * Cache team logo locally
     */
    public function cacheTeamLogo($logo_url, $team_id) {
        if (empty($logo_url) || empty($team_id)) {
            return false;
        }
        
        $upload_dir = wp_upload_dir();
        $cache_dir = $upload_dir['basedir'] . '/api-football/logos';
        
        if (!file_exists($cache_dir)) {
            wp_mkdir_p($cache_dir);
        }
        
        $file_extension = pathinfo(parse_url($logo_url, PHP_URL_PATH), PATHINFO_EXTENSION);
        $url_hash = substr(md5($logo_url), 0, 8); // Add URL hash to filename
        $local_filename = 'team_' . $team_id . '_' . $url_hash . '.' . ($file_extension ?: 'png');
        $local_path = $cache_dir . '/' . $local_filename;
        $local_url = $upload_dir['baseurl'] . '/api-football/logos/' . $local_filename;
        
        // Check if file already exists and is recent (within cache duration)
        if (file_exists($local_path)) {
            $file_time = filemtime($local_path);
            $cache_duration = $this->getOption('cache_duration', 4 * HOUR_IN_SECONDS);
            
            if ((time() - $file_time) < $cache_duration) {
                return $local_url;
            }
        }
        
        // Download and cache the image
        $response = wp_remote_get($logo_url, array('timeout' => 30));
        
        if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
            $image_data = wp_remote_retrieve_body($response);
            
            if (!empty($image_data)) {
                file_put_contents($local_path, $image_data);
                return $local_url;
            }
        }
        
        // Return original URL if caching fails
        return $logo_url;
    }
    
    /**
     * Get option with fallback
     */
    private function getOption($key, $default = '') {
        return isset($this->options[$key]) ? $this->options[$key] : $default;
    }
    
    /**
     * Clear all cached data
     */
    public function clearCache() {
        global $wpdb;
        
        // Clear transients
        $wpdb->query($wpdb->prepare("
            DELETE FROM {$wpdb->options} 
            WHERE option_name LIKE %s 
            OR option_name LIKE %s
        ", '_transient_api_football_%', '_transient_timeout_api_football_%'));
        
        // Clear backup options
        $wpdb->query($wpdb->prepare("
            DELETE FROM {$wpdb->options} 
            WHERE option_name LIKE %s
        ", 'api_football_backup_%'));
        
        // Clear cached logos
        $upload_dir = wp_upload_dir();
        $cache_dir = $upload_dir['basedir'] . '/api-football/logos';
        
        if (is_dir($cache_dir)) {
            $files = glob($cache_dir . '/*');
            foreach ($files as $file) {
                if (is_file($file)) {
                    unlink($file);
                }
            }
        }
    }
    
    /**
     * Test API connection
     */
    public function testConnection() {
        $response = $this->makeRequest('/status');
        return $response !== false;
    }
    
    /**
     * Format date according to current locale
     */
    public function formatDate($date_string, $format = 'short') {
        $locale = get_locale();
        $timestamp = strtotime($date_string);
        
        if (!$timestamp) {
            return $date_string;
        }
        
        // Serbian locale handling
        if (strpos($locale, 'sr') === 0) {
            // Serbian months
            $months = array(
                1 => __('Jan', 'api-football'),
                2 => __('Feb', 'api-football'),
                3 => __('Mar', 'api-football'),
                4 => __('Apr', 'api-football'),
                5 => __('May', 'api-football'),
                6 => __('Jun', 'api-football'),
                7 => __('Jul', 'api-football'),
                8 => __('Aug', 'api-football'),
                9 => __('Sep', 'api-football'),
                10 => __('Oct', 'api-football'),
                11 => __('Nov', 'api-football'),
                12 => __('Dec', 'api-football')
            );
            
            $day = date('j', $timestamp);
            $month = $months[intval(date('n', $timestamp))];
            $year = date('Y', $timestamp);
            
            if ($format === 'short') {
                return $day . ' ' . $month . ', ' . $year;
            } else {
                return $day . '. ' . $month . ' ' . $year . '.';
            }
        }
        
        // Default English format
        if ($format === 'short') {
            return date('M j, Y', $timestamp);
        } else {
            return date('F j, Y', $timestamp);
        }
    }
    
    /**
     * Format time according to current locale
     */
    public function formatTime($date_string) {
        // The date string from API is already in the requested timezone
        // We need to parse it correctly without timezone conversion
        
        // Extract just the time portion from the ISO 8601 date string
        // Format: 2025-08-30T16:30:00+02:00
        if (preg_match('/T(\d{2}:\d{2})/', $date_string, $matches)) {
            return $matches[1];
        }
        
        // Fallback to old method if format doesn't match
        $timestamp = strtotime($date_string);
        
        if (!$timestamp) {
            return $date_string;
        }
        
        return date('H:i', $timestamp);
    }
    
    /**
     * Apply team corrections to API response data
     */
    private function applyTeamCorrectionsToResponse($response, $league_id = null) {
        if (!isset($response['response']) || !is_array($response['response'])) {
            return $response;
        }
        
        foreach ($response['response'] as &$data) {
            // Apply corrections to standings data
            if (isset($data['league']['standings'][0])) {
                foreach ($data['league']['standings'][0] as &$standing) {
                    if (isset($standing['team'])) {
                        $standing['team'] = $this->team_corrections->applyCorrections($standing['team'], $league_id);
                    }
                }
            }
            
            // Apply corrections to fixture data
            if (isset($data['teams'])) {
                if (isset($data['teams']['home'])) {
                    $data['teams']['home'] = $this->team_corrections->applyCorrections($data['teams']['home'], $league_id);
                }
                if (isset($data['teams']['away'])) {
                    $data['teams']['away'] = $this->team_corrections->applyCorrections($data['teams']['away'], $league_id);
                }
            }
        }
        
        return $response;
    }
} 