<?php
/**
 * Plugin Name: API Football
 * Plugin URI: https://github.com/your-username/api-football-plugin
 * Description: Display football league standings, fixtures, and results using API-Football.com integration with customizable widgets and shortcodes.
 * Version: 1.2.0
 * Author: Your Name
 * Author URI: https://yourwebsite.com
 * Text Domain: api-football
 * Domain Path: /languages
 * Requires at least: 6.0
 * Tested up to: 6.8.1
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('API_FOOTBALL_VERSION', '1.2.0');
define('API_FOOTBALL_PLUGIN_URL', plugin_dir_url(__FILE__));
define('API_FOOTBALL_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('API_FOOTBALL_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main plugin class
 */
class ApiFootballPlugin {
    
    /**
     * Instance of this class
     */
    private static $instance = null;
    
    /**
     * Get instance
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        add_action('plugins_loaded', array($this, 'init'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    /**
     * Initialize plugin
     */
    public function init() {
        // Load text domain for internationalization
        add_action('init', array($this, 'loadTextdomain'));
        
        // Include required files
        $this->includeFiles();
        
        // Initialize components
        $this->initializeComponents();
        
        // Add admin hooks
        if (is_admin()) {
            add_action('admin_menu', array($this, 'addAdminMenu'));
            add_action('admin_init', array($this, 'adminInit'));
            add_action('admin_enqueue_scripts', array($this, 'adminEnqueueScripts'));
        }
        
        // Frontend hooks
        add_action('wp_enqueue_scripts', array($this, 'frontendEnqueueScripts'));
    }
    
    /**
     * Load text domain for translations
     */
    public function loadTextdomain() {
        $locale = apply_filters('plugin_locale', get_locale(), 'api-football');
        
        // Load language file from languages directory
        load_textdomain('api-football', API_FOOTBALL_PLUGIN_PATH . 'languages/api-football-' . $locale . '.mo');
        
        // Fallback to WordPress default
        load_plugin_textdomain('api-football', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }
    
    /**
     * Include required files
     */
    private function includeFiles() {
        require_once API_FOOTBALL_PLUGIN_PATH . 'includes/api-client.php';
        require_once API_FOOTBALL_PLUGIN_PATH . 'includes/shortcodes.php';
        require_once API_FOOTBALL_PLUGIN_PATH . 'includes/widgets.php';
        require_once API_FOOTBALL_PLUGIN_PATH . 'includes/team-corrections.php';
        
        if (is_admin()) {
            require_once API_FOOTBALL_PLUGIN_PATH . 'admin/admin-settings.php';
        }
    }
    
    /**
     * Initialize components
     */
    private function initializeComponents() {
        // Initialize API client
        new ApiFootballClient();
        
        // Initialize shortcodes
        new ApiFootballShortcodes();
        
        // Initialize widgets
        new ApiFootballWidgets();
    }
    
    /**
     * Add admin menu
     */
    public function addAdminMenu() {
        // Main settings page
        add_options_page(
            __('API Football Settings', 'api-football'),
            __('API Football', 'api-football'),
            'manage_options',
            'api-football-settings',
            array($this, 'adminPage')
        );
        
        // Team corrections submenu
        add_submenu_page(
            'options-general.php',
            __('API Football - Team Corrections', 'api-football'),
            __('Team Corrections', 'api-football'),
            'manage_options',
            'api-football-team-corrections',
            array($this, 'teamCorrectionsPage')
        );
    }
    
    /**
     * Admin page callback
     */
    public function adminPage() {
        if (class_exists('ApiFootballAdminSettings')) {
            $admin = new ApiFootballAdminSettings();
            $admin->displayPage();
        }
    }
    
    /**
     * Team corrections page callback
     */
    public function teamCorrectionsPage() {
        require_once API_FOOTBALL_PLUGIN_PATH . 'admin/team-corrections-admin.php';
        $admin = new ApiFootballTeamCorrectionsAdmin();
        $admin->displayPage();
    }
    
    /**
     * Admin init
     */
    public function adminInit() {
        if (class_exists('ApiFootballAdminSettings')) {
            $admin = new ApiFootballAdminSettings();
            $admin->init();
        }
    }
    
    /**
     * Enqueue admin scripts and styles
     */
    public function adminEnqueueScripts($hook) {
        if ($hook !== 'settings_page_api-football-settings') {
            return;
        }
        
        // Enqueue WordPress color picker
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        
        wp_enqueue_style(
            'api-football-admin',
            API_FOOTBALL_PLUGIN_URL . 'admin/admin-styles.css',
            array('wp-color-picker'),
            API_FOOTBALL_VERSION
        );
        
        wp_enqueue_script(
            'api-football-admin',
            API_FOOTBALL_PLUGIN_URL . 'admin/admin-scripts.js',
            array('jquery', 'wp-color-picker'),
            API_FOOTBALL_VERSION,
            true
        );
    }
    
    /**
     * Enqueue frontend scripts and styles
     */
    public function frontendEnqueueScripts() {
        wp_enqueue_style(
            'api-football-frontend',
            API_FOOTBALL_PLUGIN_URL . 'assets/css/frontend-styles.css',
            array(),
            API_FOOTBALL_VERSION
        );
        
        wp_enqueue_script(
            'api-football-frontend',
            API_FOOTBALL_PLUGIN_URL . 'assets/js/frontend-scripts.js',
            array('jquery'),
            API_FOOTBALL_VERSION,
            true
        );
        
        // Localize script for JavaScript translations
        wp_localize_script('api-football-frontend', 'apiFootballL10n', array(
            'loading' => __('Loading...', 'api-football'),
            'error' => __('Error loading data', 'api-football'),
            'noData' => __('No data available', 'api-football'),
            'refresh' => __('Refresh', 'api-football'),
            'loadMore' => __('Load More', 'api-football')
        ));
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        // Set default options
        $default_options = array(
            'api_key' => '',
            'league_id' => '286',
            'season' => '2024',
            'team_id' => '12360',
            'timezone' => 'Europe/Belgrade',
            'color_scheme' => 'blue',
            'cache_duration' => 4 * HOUR_IN_SECONDS
        );
        
        add_option('api_football_options', $default_options);
        
        // Create uploads directory for cached images
        $upload_dir = wp_upload_dir();
        $api_football_dir = $upload_dir['basedir'] . '/api-football';
        if (!file_exists($api_football_dir)) {
            wp_mkdir_p($api_football_dir);
        }
        
        // Create team corrections table
        $this->createTeamCorrectionsTable();
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }
    
    /**
     * Create team corrections database table
     */
    private function createTeamCorrectionsTable() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'api_football_team_corrections';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            team_id int NOT NULL,
            original_name varchar(255) NOT NULL,
            corrected_name varchar(255) DEFAULT NULL,
            custom_logo_url text DEFAULT NULL,
            league_id int DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY team_league (team_id, league_id),
            KEY team_id (team_id),
            KEY league_id (league_id)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clear scheduled events
        wp_clear_scheduled_hook('api_football_clear_cache');
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }
}

// Initialize plugin
ApiFootballPlugin::getInstance(); 