<?php
/**
 * API Football Admin Settings
 * Handles the admin configuration page
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Elementor colors
/**
 * Helper function to get Elementor's secondary color hex.
 * This function should ideally be in a separate helper file or your main plugin file
 * and included before your main class is instantiated.
 */
if ( ! function_exists( 'get_elementor_secondary_color_hex' ) ) {
    function get_elementor_secondary_color_hex() {
        $elementor_active_kit_id = get_option( 'elementor_active_kit' );
        if ( ! $elementor_active_kit_id ) {
            return false;
        }
        $kit_settings = get_post_meta( $elementor_active_kit_id, '_elementor_page_settings', true );
        if ( empty( $kit_settings ) || ! is_array( $kit_settings ) ) {
            return false;
        }

        if ( isset( $kit_settings['system_colors'] ) && is_array( $kit_settings['system_colors'] ) ) {
            foreach ( $kit_settings['system_colors'] as $color_item ) {
                if ( isset( $color_item['_id'] ) && $color_item['_id'] === 'secondary' && isset( $color_item['color'] ) ) {
                    return $color_item['color'];
                }
            }
        }
        // Fallback for older Elementor versions or if not found in system_colors
        if ( isset( $kit_settings['global_colors'] ) && is_array( $kit_settings['global_colors'] ) ) {
            foreach ( $kit_settings['global_colors'] as $color_item ) {
                if ( isset( $color_item['title'] ) && strtolower( $color_item['title'] ) === 'secondary' && isset( $color_item['value'] ) ) {
                    return $color_item['value'];
                }
            }
        }
        return false;
    }
}

/**
 * Helper function to get Elementor's accent color hex.
 * This function should ideally be in a separate helper file or your main plugin file
 * and included before this class is instantiated.
 */
if ( ! function_exists( 'get_elementor_accent_color_hex' ) ) {
    function get_elementor_accent_color_hex() {
        $elementor_active_kit_id = get_option( 'elementor_active_kit' );
        if ( ! $elementor_active_kit_id ) {
            return false;
        }
        $kit_settings = get_post_meta( $elementor_active_kit_id, '_elementor_page_settings', true );
        if ( empty( $kit_settings ) || ! is_array( $kit_settings ) ) {
            return false;
        }

        if ( isset( $kit_settings['system_colors'] ) && is_array( $kit_settings['system_colors'] ) ) {
            foreach ( $kit_settings['system_colors'] as $color_item ) {
                if ( isset( $color_item['_id'] ) && $color_item['_id'] === 'accent' && isset( $color_item['color'] ) ) {
                    return $color_item['color'];
                }
            }
        }
        // Fallback for older Elementor versions or if not found in system_colors
        if ( isset( $kit_settings['global_colors'] ) && is_array( $kit_settings['global_colors'] ) ) {
            foreach ( $kit_settings['global_colors'] as $color_item ) {
                if ( isset( $color_item['title'] ) && strtolower( $color_item['title'] ) === 'accent' && isset( $color_item['value'] ) ) {
                    return $color_item['value'];
                }
            }
        }
        return false;
    }
}
// End Elementor colors

class ApiFootballAdminSettings {
    
    private $options;
    
    public function __construct() {
        $this->options = get_option('api_football_options', array());
    }
    
    /**
     * Initialize admin settings
     */
    public function init() {
        register_setting(
            'api_football_settings',
            'api_football_options',
            array($this, 'validateOptions')
        );
        
        add_settings_section(
            'api_football_main_section',
            __('API Configuration', 'api-football'),
            array($this, 'sectionCallback'),
            'api_football_settings'
        );
        
        add_settings_field(
            'api_key',
            __('API Key', 'api-football'),
            array($this, 'apiKeyCallback'),
            'api_football_settings',
            'api_football_main_section'
        );
        
        add_settings_field(
            'league_id',
            __('League ID', 'api-football'),
            array($this, 'leagueIdCallback'),
            'api_football_settings',
            'api_football_main_section'
        );
        
        add_settings_field(
            'season',
            __('Season', 'api-football'),
            array($this, 'seasonCallback'),
            'api_football_settings',
            'api_football_main_section'
        );
        
        add_settings_field(
            'team_id',
            __('Default Team ID', 'api-football'),
            array($this, 'teamIdCallback'),
            'api_football_settings',
            'api_football_main_section'
        );
        
        add_settings_field(
            'timezone',
            __('Timezone', 'api-football'),
            array($this, 'timezoneCallback'),
            'api_football_settings',
            'api_football_main_section'
        );
        
        add_settings_section(
            'api_football_display_section',
            __('Display Settings', 'api-football'),
            array($this, 'displaySectionCallback'),
            'api_football_settings'
        );
        
        add_settings_field(
            'primary_color',
            __('Primary Color', 'api-football'),
            array($this, 'primaryColorCallback'),
            'api_football_settings',
            'api_football_display_section'
        );
        
        add_settings_field(
            'accent_color',
            __('Accent Color', 'api-football'),
            array($this, 'accentColorCallback'),
            'api_football_settings',
            'api_football_display_section'
        );
        
        add_settings_field(
            'cache_duration',
            __('Cache Duration (hours)', 'api-football'),
            array($this, 'cacheDurationCallback'),
            'api_football_settings',
            'api_football_display_section'
        );
    }
    
    /**
     * Display admin page
     */
    public function displayPage() {
        // Handle clear cache action
        if (isset($_POST['clear_cache']) && wp_verify_nonce($_POST['_wpnonce'], 'clear_cache')) {
            $api_client = new ApiFootballClient();
            $api_client->clearCache();
            echo '<div class="notice notice-success"><p>' . __('Cache cleared successfully!', 'api-football') . '</p></div>';
        }
        
        // Handle test connection action
        if (isset($_POST['test_connection']) && wp_verify_nonce($_POST['_wpnonce'], 'test_connection')) {
            $api_client = new ApiFootballClient();
            $connection_test = $api_client->testConnection();
            
            if ($connection_test) {
                echo '<div class="notice notice-success"><p>' . __('API connection successful!', 'api-football') . '</p></div>';
            } else {
                echo '<div class="notice notice-error"><p>' . __('API connection failed. Please check your API key.', 'api-football') . '</p></div>';
            }
        }
        ?>
        <div class="wrap">
            <h1><?php _e('API Football Settings', 'api-football'); ?></h1>
            
            <form method="post" action="options.php">
                <?php
                settings_fields('api_football_settings');
                do_settings_sections('api_football_settings');
                submit_button();
                ?>
            </form>
            
            <hr>
            
            <h2><?php _e('Tools', 'api-football'); ?></h2>
            
            <div class="api-football-tools">
                <form method="post" style="display: inline-block; margin-right: 10px;">
                    <?php wp_nonce_field('test_connection'); ?>
                    <input type="submit" name="test_connection" class="button button-secondary" value="<?php _e('Test API Connection', 'api-football'); ?>">
                </form>
                
                <form method="post" style="display: inline-block;">
                    <?php wp_nonce_field('clear_cache'); ?>
                    <input type="submit" name="clear_cache" class="button button-secondary" value="<?php _e('Clear Cache', 'api-football'); ?>" onclick="return confirm('<?php _e('Are you sure you want to clear all cached data?', 'api-football'); ?>');">
                </form>
            </div>
            
            <hr>
            
            <h2><?php _e('Shortcode Examples', 'api-football'); ?></h2>
            
            <div class="api-football-shortcodes">
                <h3><?php _e('Homepage Widgets', 'api-football'); ?></h3>
                <p><strong><?php _e('Standings Table (Compact):', 'api-football'); ?></strong></p>
                <code>[api_football_standings limit="10"]</code>
                
                <p><strong><?php _e('Next 5 Fixtures:', 'api-football'); ?></strong></p>
                <code>[api_football_fixtures next="5"]</code>
                
                <p><strong><?php _e('Last 5 Results:', 'api-football'); ?></strong></p>
                <code>[api_football_results last="5"]</code>
                
                <h3><?php _e('Full Page Widgets', 'api-football'); ?></h3>
                <p><strong><?php _e('Full Standings Table:', 'api-football'); ?></strong></p>
                <code>[api_football_standings_full]</code>
                
                <p><strong><?php _e('Full Schedule:', 'api-football'); ?></strong></p>
                <code>[api_football_schedule]</code>
                
                <p><strong><?php _e('Full Results:', 'api-football'); ?></strong></p>
                <code>[api_football_results_full]</code>
                
                <h3><?php _e('Custom Parameters', 'api-football'); ?></h3>
                <p><?php _e('You can override default settings using these parameters:', 'api-football'); ?></p>
                <ul>
                    <li><code>league="286"</code> - <?php _e('League ID', 'api-football'); ?></li>
                    <li><code>season="2024"</code> - <?php _e('Season year', 'api-football'); ?></li>
                    <li><code>team="12360"</code> - <?php _e('Team ID (for fixtures/results)', 'api-football'); ?></li>
                </ul>
                
                <p><strong><?php _e('Example with custom parameters:', 'api-football'); ?></strong></p>
                <code>[api_football_standings league="286" season="2024" limit="20"]</code>
            </div>
        </div>
        <?php
    }
    
    /**
     * Section callbacks
     */
    public function sectionCallback() {
        echo '<p>' . __('Configure your API-Football.com settings below.', 'api-football') . '</p>';
    }
    
    public function displaySectionCallback() {
        echo '<p>' . __('Customize the appearance and behavior of your football widgets.', 'api-football') . '</p>';
    }
    
    /**
     * Field callbacks
     */
    public function apiKeyCallback() {
        $value = isset($this->options['api_key']) ? $this->options['api_key'] : '21520870122bb16d77ddcffa5d31e225';
        echo '<input type="text" name="api_football_options[api_key]" value="' . esc_attr($value) . '" class="regular-text" />';
        echo '<p class="description">' . __('Your API key from API-Football.com', 'api-football') . '</p>';
    }
    
    public function leagueIdCallback() {
        $value = isset($this->options['league_id']) ? $this->options['league_id'] : '286';
        echo '<input type="number" name="api_football_options[league_id]" value="' . esc_attr($value) . '" class="small-text" />';
        echo '<p class="description">' . __('Default league ID (286 = Serbian Super Liga)', 'api-football') . '</p>';
    }
    
    public function seasonCallback() {
        $value = isset($this->options['season']) ? $this->options['season'] : '2024';
        echo '<input type="number" name="api_football_options[season]" value="' . esc_attr($value) . '" class="small-text" min="2000" max="2030" />';
        echo '<p class="description">' . __('Default season year', 'api-football') . '</p>';
    }
    
    public function teamIdCallback() {
        $value = isset($this->options['team_id']) ? $this->options['team_id'] : '12360';
        echo '<input type="number" name="api_football_options[team_id]" value="' . esc_attr($value) . '" class="small-text" />';
        echo '<p class="description">' . __('Default team ID for fixtures/results (12360 = Železničar Pančevo)', 'api-football') . '</p>';
    }
    
    public function timezoneCallback() {
        $value = isset($this->options['timezone']) ? $this->options['timezone'] : 'Europe/Belgrade';
        $timezones = array(
            'Europe/Belgrade' => 'Europe/Belgrade (UTC+1)',
            'Europe/Sofia' => 'Europe/Sofia (UTC+2)',
            'UTC' => 'UTC (GMT+0)',
            'Europe/London' => 'Europe/London (GMT+0)',
            'Europe/Paris' => 'Europe/Paris (UTC+1)',
            'Europe/Berlin' => 'Europe/Berlin (UTC+1)',
            'Europe/Rome' => 'Europe/Rome (UTC+1)',
            'Europe/Madrid' => 'Europe/Madrid (UTC+1)',
            'Europe/Amsterdam' => 'Europe/Amsterdam (UTC+1)'
        );
        
        echo '<select name="api_football_options[timezone]">';
        foreach ($timezones as $tz => $label) {
            $selected = selected($value, $tz, false);
            echo '<option value="' . esc_attr($tz) . '"' . $selected . '>' . esc_html($label) . '</option>';
        }
        echo '</select>';
        echo '<p class="description">' . __('Timezone for displaying match times', 'api-football') . '</p>';
    }
	
	public function primaryColorCallback() {
		$elementor_secondary = get_elementor_secondary_color_hex();
        $default_color = $elementor_secondary ? $elementor_secondary : '#4A90E2'; // Fallback to your original if Elementor not found
        $value = isset($this->options['primary_color']) ? $this->options['primary_color'] : $default_color;
        echo '<input type="text" name="api_football_options[primary_color]" value="' . esc_attr($value) . '" class="api-football-color-picker" />';
        echo '<p class="description">' . __('Delete the color, then click Save to obtain the Elementor secondary color', 'api-football') . '</p>';
    }
    
//     public function primaryColorCallback() {
//         $value = isset($this->options['primary_color']) ? $this->options['primary_color'] : '#4A90E2';
//         echo '<input type="text" name="api_football_options[primary_color]" value="' . esc_attr($value) . '" class="api-football-color-picker" />';
//         echo '<p class="description">' . __('Primary color for headers, buttons, and main elements', 'api-football') . '</p>';
//     }

	public function accentColorCallback() {
		$elementor_accent = get_elementor_accent_color_hex();
		$default_color = $elementor_accent ? $elementor_accent : '#2196F3'; // Fallback to your original if Elementor not found
        $value = isset($this->options['accent_color']) ? $this->options['accent_color'] : $default_color;
        echo '<input type="text" name="api_football_options[accent_color]" value="' . esc_attr($value) . '" class="api-football-color-picker" />';
        echo '<p class="description">' . __('Delete the color, then click Save to obtain the Elementor accent color', 'api-football') . '</p>';
    }
    
//     public function accentColorCallback() {
//         $value = isset($this->options['accent_color']) ? $this->options['accent_color'] : '#2196F3';
//         echo '<input type="text" name="api_football_options[accent_color]" value="' . esc_attr($value) . '" class="api-football-color-picker" />';
//         echo '<p class="description">' . __('Accent color for highlights, links, and secondary elements', 'api-football') . '</p>';
//     }
    
    public function cacheDurationCallback() {
        $value = isset($this->options['cache_duration']) ? $this->options['cache_duration'] / HOUR_IN_SECONDS : 4;
        echo '<input type="number" name="api_football_options[cache_duration]" value="' . esc_attr($value) . '" class="small-text" min="1" max="24" />';
        echo '<p class="description">' . __('How long to cache API responses (1-24 hours)', 'api-football') . '</p>';
    }
    
    /**
     * Validate options
     */
    public function validateOptions($input) {
        $output = array();
        
        if (isset($input['api_key'])) {
            $output['api_key'] = sanitize_text_field($input['api_key']);
        }
        
        if (isset($input['league_id'])) {
            $output['league_id'] = absint($input['league_id']);
        }
        
        if (isset($input['season'])) {
            $season = absint($input['season']);
            if ($season >= 2000 && $season <= 2030) {
                $output['season'] = $season;
            }
        }
        
        if (isset($input['team_id'])) {
            $output['team_id'] = absint($input['team_id']);
        }
        
        if (isset($input['timezone'])) {
            $output['timezone'] = sanitize_text_field($input['timezone']);
        }
        
        if (isset($input['primary_color'])) {
            $color = sanitize_hex_color($input['primary_color']);
            if ($color) {
                $output['primary_color'] = $color;
            }
        }
        
        if (isset($input['accent_color'])) {
            $color = sanitize_hex_color($input['accent_color']);
            if ($color) {
                $output['accent_color'] = $color;
            }
        }
        
        if (isset($input['cache_duration'])) {
            $hours = absint($input['cache_duration']);
            if ($hours >= 1 && $hours <= 24) {
                $output['cache_duration'] = $hours * HOUR_IN_SECONDS;
            }
        }
        
        return $output;
    }
} 